#!/bin/bash

# Copyright (c) 2022-2023 Vossloh-Schwabe Deutschland GmbH
# Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the “Software”), to deal
# in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
# the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
# The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
# WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
# OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
# OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.


LANGW_WHL="install_files/vs_lan_gateway_server-2.0-py3-none-any.whl"

SCRIPT_PATH=$(dirname "$(realpath -s "$0")")
cd $SCRIPT_PATH
err () {
    echo "Error in $(caller)"
    exit 1
}
trap 'err' err
if [[ $EUID -ne 0 ]]; then
    echo "Has to be run as root"
    exit 1
fi

join_arr() {
  local IFS="$1"
  shift
  echo "$*"
}

stringContain() {
    [ -z "$1" ] || { [ -z "${2##*$1*}" ] && [ -n "$2" ];};
}

service_exists() {
    local n=$1
    if [[ $(systemctl list-units --all -t service --full --no-legend "$n.service" | sed 's/^\s*//g' | cut -f1 -d' ') == $n.service ]]; then
        return 0
    else
        return 1
    fi
}

config_path_old=/root/.vs_lan_gateway_server
config_file_old=/root/.vs_lan_gateway_server/config.json
config_path_new=/var/vs_lan_gateway_server
config_file_new=/var/vs_lan_gateway_server/config.json


echo 'Check for running lan-gateway service'
if service_exists lan-gateway; then
    echo 'Stopping the running lan-gateway service'
    /bin/systemctl stop lan-gateway
fi

if [ ! -e $config_file_old ] && [ ! -e $config_file_new ]; then
    echo ' '
    echo ' '
    echo '#########################################################'
    echo '# An Update is not possible.                            #'
    echo '# The Configuration file could not found.               #'
    echo '# Please use "setup.sh" to create new config instead.   #'    
    echo '#########################################################'
    exit 1
elif [ ! -e $config_file_new ]; then
    mkdir -p $config_path_new
    mkdir -p $config_path_new/dashboards
    chmod 755 $config_path_new
    chmod 755 $config_path_new/dashboards   
fi  

if [ -e $config_file_old ] && [ ! -e $config_file_new ]; then
    echo 'Copying existing config to new location $config_path_new'
    cp $config_file_old $config_path_new
fi


VERSION=$(sed 's/\..*//' /etc/debian_version)

if [ $VERSION = 11 ]; then
	curl -O http://http.us.debian.org/debian/pool/main/d/debian-archive-keyring/debian-archive-keyring_2023.4_all.deb
	dpkg -i debian-archive-keyring_2023.4_all.deb	
	echo "deb http://deb.debian.org/debian bullseye-backports main" | sudo tee /etc/apt/sources.list.d/bullseye-backports.list
fi


if [ $dbms = "influxdb" ]; then
	if [ -e /etc/apt/sources.list.d/influxdb.list ]; then
		rm -f /usr/share/keyrings/influxdb-archive-keyring.gpg
		rm -f /etc/apt/sources.list.d/influxdb.list
	fi
    curl https://repos.influxdata.com/influxdb.key | gpg --dearmor | sudo tee /usr/share/keyrings/influxdb-archive-keyring.gpg >/dev/null
    echo "deb [signed-by=/usr/share/keyrings/influxdb-archive-keyring.gpg] https://repos.influxdata.com/debian $(lsb_release -cs) stable" | sudo tee /etc/apt/sources.list.d/influxdb.list
fi

if [ -e /etc/apt/sources.list.d/grafana.list ]; then
	rm -f /usr/share/keyrings/grafana-archive-keyrings.gpg
	rm -f /etc/apt/sources.list.d/grafana.list
fi	
curl https://packages.grafana.com/gpg.key | gpg --dearmor | sudo tee /usr/share/keyrings/grafana-archive-keyrings.gpg >/dev/null
echo "deb [signed-by=/usr/share/keyrings/grafana-archive-keyrings.gpg] https://packages.grafana.com/oss/deb stable main" | sudo tee /etc/apt/sources.list.d/grafana.list

echo 'Update Raspberry Pi OS'
apt-get update
apt-get -y upgrade

echo 'Install neccessary Packages'
apt-get install -y jq python3-pip python3-werkzeug adduser libfontconfig1 libssl-dev grafana
if [ $VERSION = 11 ]; then
	apt-get -t bullseye-backports install -y pipx
else
	apt-get install -y pipx
fi
if [ -d /usr/local/lib/python3.9/dist-packages/sslpsk2 ]; then
	echo 'Remove old SSLPSK2 package'
	pip uninstall -y sslpsk2
fi

if [ -d /usr/local/lib/python3.9/dist-packages/vs_lan_gateway_server ]; then
	echo 'Remove old VS LAN GAteway Server package'
	pip uninstall -y vs_lan_gateway_server
fi
echo 'Install new VS LAN Gateway Server Demo package'
PIPX_HOME=/opt/pipx PIPX_BIN_DIR=/usr/local/bin pipx install --force $LANGW_WHL

echo 'Setting up lan-gateway service configuration file'
cp install_files/lan-gateway.service /etc/systemd/system/lan-gateway.service
chmod 644 /etc/systemd/system/lan-gateway.service

cp install_files/dashboard.yaml /etc/grafana/provisioning/dashboards/
grafana-cli plugins install ryantxu-ajax-panel
echo 'Starting Grafana'
/bin/systemctl daemon-reload
/bin/systemctl enable grafana-server
/bin/systemctl start grafana-server

echo ' '
echo ' '
echo '################################################################################################'
echo '# Update successful.                                                                           #'
echo '# To enable the server start at boot, run "sudo systemctl enable lan-gateway"                  #'
echo '# To start the server manually or now, run "sudo systemctl start lan-gateway"                  #'
echo '# The configuration website is still reachable at following addresses:                         #'
for a in `hostname -I`; do
    if stringContain ':' $a; then
        echo "  http://[$a]:31460/"
    else
        echo "  http://$a:31460/"
    fi
done
echo '################################################################################################'
