#!/bin/bash

# Copyright (c) 2022-2023 Vossloh-Schwabe Deutschland GmbH
# Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the “Software”), to deal 
# in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of 
# the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
# The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
# WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
# OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
# OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.


LANGW_WHL="install_files/vs_lan_gateway_server-2.0-py3-none-any.whl"

SCRIPT_PATH=$(dirname "$(realpath -s "$0")")
cd $SCRIPT_PATH
err () {
    echo "Error in $(caller)"
    exit 1
}
trap 'err' err
if [[ $EUID -ne 0 ]]; then
    echo "Has to be run as root"
    exit 1
fi

join_arr() {
    local IFS="$1"
    shift
    echo "$*"
}

stringContain() {
    [ -z "$1" ] || { [ -z "${2##*$1*}" ] && [ -n "$2" ];};
}

service_exists() {
    local n=$1
    if [[ $(systemctl list-units --all -t service --full --no-legend "$n.service" | sed 's/^\s*//g' | cut -f1 -d' ') == $n.service ]]; then
        return 0
    else
        return 1
    fi
}

json=""
dbms=none
config_file=/var/vs_lan_gateway_server/config.json
config_path_old=/root/.vs_lan_gateway_server
config_file_old=/root/.vs_lan_gateway_server/config.json
config_path_new=/var/vs_lan_gateway_server
config_file_new=/var/vs_lan_gateway_server/config.json

echo 'Check for running lan-gateway service'
if service_exists lan-gateway; then
	echo 'Stopping the running lan-gateway service'
    /bin/systemctl stop lan-gateway
fi

if [ -e $config_file_old ] || [ -e $config_file_new ]; then
    echo 'The Configuration file is already existing. You can use "update.sh" to keep it'
    config_answer=o
    while [ $config_answer != "n" -a $config_answer != "y" ]; do
        read -p "Do you really want to create new configuration? ((y)es/(n)o): " config_answer
    done
    if [ $config_answer = "n" ]; then
        echo ' '
        echo ' '
        echo '###########################################################'
        echo '# Setup stopped.                                          #'
        echo '# Please use "update.sh" to update without config change. #'
        echo '###########################################################'
        exit 1
    fi
fi

mkdir -p $config_path_new

echo 'Database type selection'

while [ $dbms != "influxdb" -a $dbms != "mariadb" -a $dbms != "i" -a $dbms != "m" ]; do
    read -p "Enter the database system you want to use ((i)nfluxdb/(m)ariadb): " dbms
done

if [ $dbms = "i" ]; then
    dbms=influxdb
    packages=$dbms
elif [ $dbms = "m" ]; then
    dbms=mariadb
    packages="mariadb-server libmariadb-dev"
fi

mkdir -p /etc/grafana/provisioning/datasources
mkdir -p /etc/grafana/provisioning/dashboards

VERSION=$(sed 's/\..*//' /etc/debian_version)
if [ $VERSION = 11 ]; then
	curl -O http://http.us.debian.org/debian/pool/main/d/debian-archive-keyring/debian-archive-keyring_2023.4_all.deb
	dpkg -i debian-archive-keyring_2023.4_all.deb	
	echo "deb http://deb.debian.org/debian bullseye-backports main" | sudo tee /etc/apt/sources.list.d/bullseye-backports.list
fi

if [ $dbms = "influxdb" ]; then
	if [ -e /etc/apt/sources.list.d/influxdb.list ]; then
		sudo rm -f /usr/share/keyrings/influxdb-archive-keyring.gpg
		sudo rm -f /etc/apt/sources.list.d/influxdb.list
	fi
    curl https://repos.influxdata.com/influxdb.key | gpg --dearmor | sudo tee /usr/share/keyrings/influxdb-archive-keyring.gpg >/dev/null
    echo "deb [signed-by=/usr/share/keyrings/influxdb-archive-keyring.gpg] https://repos.influxdata.com/debian $(lsb_release -cs) stable" | sudo tee /etc/apt/sources.list.d/influxdb.list
fi

if [ -e /etc/apt/sources.list.d/grafana.list ]; then
	sudo rm -f /usr/share/keyrings/grafana-archive-keyrings.gpg
	sudo rm -f /etc/apt/sources.list.d/grafana.list
fi	
curl https://packages.grafana.com/gpg.key | gpg --dearmor | sudo tee /usr/share/keyrings/grafana-archive-keyrings.gpg >/dev/null
echo "deb [signed-by=/usr/share/keyrings/grafana-archive-keyrings.gpg] https://packages.grafana.com/oss/deb stable main" | sudo tee /etc/apt/sources.list.d/grafana.list

echo 'Update Raspberry Pi OS'
apt update
apt -y upgrade
echo 'Install neccessary Packages'
apt install -y jq python3-pip python3-werkzeug adduser libfontconfig1 libssl-dev $packages grafana
if [ $VERSION = 11 ]; then
	apt-get -t bullseye-backports install -y pipx
else
	apt-get install -y pipx
fi

if [ -d /usr/local/lib/python3.9/dist-packages/sslpsk2 ]; then
	echo 'Remove old SSLPSK2 package'
	pip uninstall -y sslpsk2
fi

if [ -d /usr/local/lib/python3.9/dist-packages/vs_lan_gateway_server ]; then
	echo 'Remove old VS LAN Gateway Server Demo package'
	pip uninstall -y vs_lan_gateway_server
fi
echo 'Install new VS LAN Gateway Server Demo package'
PIPX_HOME=/opt/pipx PIPX_BIN_DIR=/usr/local/bin pipx install $LANGW_WHL

if [ $dbms = "mariadb" ]; then
	while
		read -p "Enter the database name (no spaces allowed): " -e -i "blu2light" mariadb_dbname
		[[ $mariadb_dbname == *' '* ]]
	do echo "Enter a valid database name!"; done
	while
		read -p "Enter the database username ('root' or spaces in name are not allowed): " -e -i "b2l_user" mariadb_username
		[[ $mariadb_username == "root" ]] || [[ $mariadb_username == *' '* ]]
	do echo "Enter a valid database username!"; done
    read -rsp "Enter the database password: " mariadb_password
	pw_js=`echo -n "$mariadb_password" | jq -Rsa .`
    cat ./install_files/mariadb.yaml | sed -e "s/<database>/$mariadb_dbname/" -e "s/<username>/$mariadb_username/" -e "s/<password>/$pw_js/" > /etc/grafana/provisioning/datasources/mariadb.yaml
    echo
    json+="{\"mariadb\": {\"enabled\": true, \"username\": \"$mariadb_username\", \"dbname\": \"$mariadb_dbname\", \"password\": $pw_js}"
    echo 'Configuring MariaDB database system'
    MY_CNF_CREATED=0
    if [ $mariadb_dbname ]; then
        if [ -e ~/.my.cnf ]; then
            MY_CNF_CREATED=1
            mv ~/.my.cnf ~/.my.cnf.000001
        fi
        cat > ~/.my.cnf <<EOF
[mysql]
user=root
password=
EOF
        cat install_files/init.sql | sed -e "s/<database>/$mariadb_dbname/" -e "s/<username>/$mariadb_username/" -e "s/<password>/$mariadb_password/" | mysql
        rm ~/.my.cnf
        if [ $MY_CNF_CREATED -eq 1 ]; then
            mv ~/.my.cnf.000001 ~/.my.cnf
        fi
    fi  
else
    cp install_files/influxdb.yaml /etc/grafana/provisioning/datasources/
fi

echo 'Database Setup done'

echo 'Please enter credentials for the web configuration'
read -p 'Username: ' web_username
read -sp 'Password: ' web_password

web_password=`python3 -c "from werkzeug.security import generate_password_hash; print(generate_password_hash(\"$web_password\"))"`
echo

if [ ! -z "$json" ]; then
    json+=", "
else
    json+="{"
fi
json+="\"web\": {\"username\": \"$web_username\", \"password\": \"$web_password\"}"

json+="}"
echo $json > $config_file_new

echo 'Setting up lan-gateway service configuration file'
cp install_files/lan-gateway.service /etc/systemd/system/lan-gateway.service
chmod 644 /etc/systemd/system/lan-gateway.service

grafana-cli plugins install ryantxu-ajax-panel

cp install_files/dashboard.yaml /etc/grafana/provisioning/dashboards/

mkdir -p $config_path_new/dashboards
chmod 755 $config_path_new
chmod 755 $config_path_new/dashboards

echo 'Starting Grafana'
/bin/systemctl daemon-reload
/bin/systemctl enable grafana-server
/bin/systemctl start grafana-server

echo ' '
echo ' '
echo '################################################################################################'
echo '# Setup successful.                                                                            #'
echo '# To enable the server start at boot, run "sudo systemctl enable lan-gateway"                  #'
echo '# To start the server manually or now, run "sudo systemctl start lan-gateway"                  #'
echo '# The configuration website is reachable at following addresses:                               #'
for a in `hostname -I`; do
    if stringContain ':' $a; then
        echo "  http://[$a]:31460/"
    else
        echo "  http://$a:31460/"
    fi
done
echo '# Some additional configurations on the configuration web page are nessesary                   #'
echo '# before the server will work completely.                                                      #'
echo '################################################################################################'
